<?php
include_once includes('config.php');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Perform input validation and ensure the user is authorized to save a card
    if (is_csrf_valid() && empty($_POST['honeypot'])) {
        // Get user ID from a secure source, e.g., a validated session
        $user_id = $_SESSION['user']['id'];

        // Sanitize and validate user input
        $cardNumber = filter_input(INPUT_POST, 'card-number', FILTER_SANITIZE_STRING);
        $cardNumber = str_replace(' ', '', $cardNumber); // Remove spaces from the card number
        $expirationMonth = (int)$_POST['expiration-month'];
        $expirationYear = filter_input(INPUT_POST, 'expiration-year', FILTER_VALIDATE_INT);
        $cvv = filter_input(INPUT_POST, 'cvv', FILTER_SANITIZE_STRING);
        $cardholderName = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_STRING);

        // Check for empty fields
        if (empty($cardNumber)) {
            $response = array('success' => false, 'message' => 'Card number is required', 'icon' => 'error');
            echo json_encode($response);
            exit;
        }
        if (empty($expirationMonth) || empty($expirationYear)) {
            $response = array('success' => false, 'message' => 'Expiration date is required', 'icon' => 'error');
            echo json_encode($response);
            exit;
        }
        if (empty($cvv)) {
            $response = array('success' => false, 'message' => 'CVV is required', 'icon' => 'error');
            echo json_encode($response);
            exit;
        }
        if (empty($cardholderName)) {
            $response = array('success' => false, 'message' => 'Cardholder name is required', 'icon' => 'error');
            echo json_encode($response);
            exit;
        }

        // Ensure that expiration date is correctly formatted (MM/YY)
        $expirationDate = sprintf('%02d/%02d', $expirationMonth, $expirationYear);

        // Validate credit card number using the Luhn algorithm
        if (!is_valid_credit_card($cardNumber)) {
            $response = array('success' => false, 'message' => 'Invalid credit card number', 'icon' => 'error');
            echo json_encode($response);
            exit; // Stop further execution
        }

        // Use prepared statement to prevent SQL injection
        $sql = "INSERT INTO credit_cards (card_number, expiration_date, cvv, cardholder_name, user_id) VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssssi", $cardNumber, $expirationDate, $cvv, $cardholderName, $user_id);

        if ($stmt->execute()) {
            $response = array('success' => true, 'message' => 'Card saved successfully', 'icon' => 'success');
        } else {
            $response = array('success' => false, 'message' => 'Error saving the card', 'icon' => 'error');
        }

        echo json_encode($response);

        $stmt->close();
        $conn->close();
    } else {
        echo "Invalid CSRF token or honeypot field is not empty.";
    }
}

// Function to validate credit card number using Luhn algorithm
function is_valid_credit_card($number) {
    $number = preg_replace('/\D/', '', $number);
    $sum = 0;
    $length = strlen($number);
    for ($i = 0; $i < $length; $i++) {
        $digit = (int)$number[$length - $i - 1];
        if ($i % 2 == 1) {
            $digit *= 2;
            if ($digit > 9) {
                $digit -= 9;
            }
        }
        $sum += $digit;
    }
    return $sum % 10 == 0;
}
?>
