<?php
// Include necessary files and start session if required
include_once includes('config.php');
include_once includes('mailer/send.php');

// Define the reminder intervals in days
$reminderIntervals = [7, 3, 1, 0]; // 0 for the last day

// Calculate the date range for searching orders due for renewal
$sevenDaysLater = date('Y-m-d', strtotime('+7 days'));
$today = date('Y-m-d');

// Query to fetch orders due for renewal in the next 7 days
$query = "SELECT * FROM orders WHERE renewal_date BETWEEN ? AND ?";
$stmt = mysqli_prepare($conn, $query);

if ($stmt) {
    mysqli_stmt_bind_param($stmt, "ss", $today, $sevenDaysLater);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);

    while ($row = mysqli_fetch_assoc($result)) {
        $orderNumber = $row['order_number'];
        $expirationDate = $row['renewal_date'];
        $dateDifference = date_diff(new DateTime($today), new DateTime($expirationDate))->days;
        $userId = $row['user_id'];
        $productId = $row['product_id'];

        // Check if an invoice already exists for the order
        if (!invoiceExistsForOrder($orderNumber, $conn)) {
            // Create an invoice for the order if it does not exist
            $totalAmount = $row['total_amount'];
            $currency = $row['currency'];
            $taxPercent = 20; // 20% tax rate
            $invoiceNumber = generateInvoiceNumber();

            createInvoice($userId, $orderNumber, $totalAmount, $currency, $invoiceNumber, $taxPercent, $conn);
            sendRenewalEmail($userId, $orderNumber, $invoiceNumber, $conn, 7, $productId); // Function to send email
            // Debug logging
            error_log("Email sent for order: $orderNumber");
        } else {
            // Invoice already exists, fetch the invoice number
            $invoiceNumber = getInvoiceNumberByOrderNumber($orderNumber, $conn);
        }

        // Ensure we have an invoice number before proceeding to send emails
// Ensure we have an invoice number before proceeding to send emails
        if ($invoiceNumber) {
            // Logic to send email reminders based on the expiration date
            if ($dateDifference == 3) {
                // Send a reminder email 3 days before the expiration
                sendRenewalEmail($userId, $orderNumber, $invoiceNumber, $conn, 3, $productId);
            } elseif ($dateDifference == 1) {
                // Send a reminder email 1 day before the expiration
                sendRenewalEmail($userId, $orderNumber, $invoiceNumber, $conn, 1, $productId);
            } elseif ($dateDifference == 0) {
                // Send a last day reminder email
                sendRenewalEmail($userId, $orderNumber, $invoiceNumber, $conn, 0, $productId);
            }
        }
        else {
            // Log an error or handle cases where the invoice number couldn't be retrieved
            error_log("Failed to retrieve invoice number for order: $orderNumber");
        }

    }

    mysqli_free_result($result);
    mysqli_stmt_close($stmt);
} else {
    // Handle database error
    error_log("Failed to prepare statement: " . mysqli_error($conn));
}

mysqli_close($conn);

// Function to check if an invoice already exists for an order
function invoiceExistsForOrder($orderNumber, $conn) {
    $stmt = $conn->prepare("SELECT COUNT(*) AS count FROM invoices WHERE JSON_CONTAINS(order_numbers, JSON_QUOTE(?)) AND invoice_status = 'Unpaid'");
    $stmt->bind_param("s", $orderNumber);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $count = $row['count'];
    $stmt->close();
    return $count > 0;
}
function getInvoiceNumberByOrderNumber($orderNumber, $conn)
{
    $stmt = $conn->prepare("SELECT invoice_number FROM invoices WHERE JSON_CONTAINS(order_numbers, JSON_QUOTE(?)) AND invoice_status = 'Unpaid'");
    $stmt->bind_param("s", $orderNumber);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $invoiceNumber = $row['invoice_number'];
    $stmt->close();
    return $invoiceNumber;
}
// Function to generate invoice number (Replace with your logic)
function generateInvoiceNumber($length = 20) {
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $orderNumber = '';

    // Add a timestamp (optional)
    $orderNumber .= date('YmdHis');

    // Add random characters to reach the desired length
    while (strlen($orderNumber) < $length) {
        $orderNumber .= $characters[rand(0, strlen($characters) - 1)];
    }

    return $orderNumber;
}

// Function to create invoice
// Function to create invoice
function createInvoice($userId, $orderNumber, $totalAmount, $currency, $invoiceNumber, $taxPercent, $conn) {
    $invoiceDate = date('Y-m-d H:i:s');

    // Convert the order number to a JSON array format
    $orderNumbersJson = json_encode([$orderNumber]);

    // Calculate tax amount based on the total amount and tax percent
    $taxAmount = ($totalAmount * $taxPercent) / 100;
    $amount = $totalAmount;
    $totalAmount = $totalAmount + ($totalAmount * 0.20); // Add 20% to the total amount

    // Insert the invoice into the database
    $sql = "INSERT INTO invoices (invoice_number, user_id, order_numbers, total_amount, currency, tax_percent, tax_amount, invoice_date, amount)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sssssssss", $invoiceNumber, $userId, $orderNumbersJson, $totalAmount, $currency, $taxPercent, $taxAmount, $invoiceDate, $amount);
    if ($stmt->execute()) {
        // Log success or debugging information
    } else {
        // Log the error message for debugging
    }
    return $invoiceNumber;
}



// Function to send renewal email to the user
function sendRenewalEmail($userId, $orderNumber, $invoiceNumber, $conn, $daysRemaining, $productId) {
    // Retrieve user email and name
    $user = getUserById($userId, $conn); // Function to retrieve user details
    $email = $user['email'];
    $name = $user['name'];
    $preferredRegion = $user['preferred_region'];

    // Get the product name based on the preferred region
    $productName = getProductName($productId, $preferredRegion, $conn);

    // Retrieve additional data from the orders table if available
    $additionalData = getAdditionalDataFromOrder($orderNumber, $conn);

    $additionalInfo = '';
    if (!empty($additionalData)) {
        $additionalInfo = ' (' . $additionalData . ')';
    }

    // Construct the service type including additional info
    $serviceType = $productName . $additionalInfo;

    // Construct email content
    $accountPrefix = ($preferredRegion === 'tr') ? '/tr' : ''; // Add '/tr' prefix if preferred region is 'tr'
    $btnLink = url("$accountPrefix/account/billing/invoices/$invoiceNumber");

    $emailContent = getEmailContent($name, $serviceType, $btnLink, $orderNumber, $preferredRegion, $daysRemaining);

    // Send email using your existing email sending mechanism
    sendEmail($email, $emailContent['subject'], $emailContent['content'], NULL);
}


function getEmailContent($name, $serviceType, $btnLink, $orderNumber, $preferredRegion, $daysRemaining) {
    require includes('language/'.$preferredRegion.'.php');

    $htmlContent = @file_get_contents(includes('emails/system_email.php'));
    if (!$htmlContent) {
        error_log("Failed to load email template for order $orderNumber");
        return [
            'subject' => 'Email template error',
            'content' => ''
        ];
    }

// Determine if it's the last day for renewal
    if ($daysRemaining == 0) {
        $daysText = sprintf($lang['renewal-email-p1-lastday'], $serviceType, $orderNumber); // For the last day
        $emailSubject = sprintf($lang['renewal-email-subject-lastday'], $serviceType, $daysRemaining);
        $headline = $lang['renewal-email-headline-lastday'];
    } elseif ($daysRemaining >= 1) {
        $daysText = sprintf($lang['renewal-email-p1'], $serviceType, $daysRemaining); // For days other than the last
        $emailSubject = sprintf($lang['renewal-email-subject'], $serviceType, $daysRemaining);
        $headline = sprintf($lang['renewal-email-headline'], $serviceType, $daysRemaining);
    }


    // Replace placeholders with dynamic content
    $replacements = [
        '{{cta-link}}' => $btnLink,
        '{{user-name}}' => $name,
        '{{headline}}' => $headline,
        '{{email-p1}}' => $daysText,
        '{{cta}}' => $lang['renewal-email-cta'] ?? '',
        '{{email-p2}}' => $lang['renewal-email-p2'] ?? '',
        '{{email-p3}}' => $lang['renewal-email-p3'] ?? '',

        // Language-specific replacements
        '{{hello}}' => $lang['email-hello'] ?? '',
        '{{email-copyright}}' => $lang['email-copyright'] ?? '',
        '{{best-regards}}' => $lang['best-regards'] ?? 'Best regards',
        '{{company-team}}' => $lang['company-team'] ?? 'The Datanito Team',
        '{{home}}' => $lang['email-home'] ?? 'Home',
        '{{about}}' => $lang['email-about'] ?? 'About',
        '{{products}}' => $lang['email-ai'] ?? 'Datanito AI',
        '{{contact}}' => $lang['email-contact'] ?? 'Contact',
        '{{privacy}}' => $lang['email-privacy'] ?? 'Privacy',
        '{{company-name}}' => $lang['company-name'] ?? 'Datanito Ltd.',
        '{{company-tagline}}' => $lang['company-tagline'] ?? 'Leading AI-Powered Software and Payment Solutions'
    ];

    foreach ($replacements as $placeholder => $value) {
        $htmlContent = str_replace($placeholder, $value, $htmlContent);
    }

    // Fill in the order number and service type dynamically in the headline if needed
    $htmlContent = sprintf($htmlContent, $orderNumber, $serviceType);

    return [
        'subject' => $emailSubject,
        'content' => $htmlContent
    ];
}
function getProductName($productId, $preferredRegion, $conn) {
    $productName = '';
    $regionColumn = ($preferredRegion === 'tr') ? 'name_tr' : 'name'; // Column name based on region

    // Prepare and execute the query to fetch the product name
    $stmt = $conn->prepare("SELECT $regionColumn FROM products WHERE product_id = ?");
    $stmt->bind_param("i", $productId);
    $stmt->execute();
    $stmt->bind_result($productName);
    $stmt->fetch();
    $stmt->close();

    return $productName;
}

// Function to retrieve additional data from orders table
function getAdditionalDataFromOrder($orderNumber, $conn) {
    $additionalData = '';
    $stmt = $conn->prepare("SELECT name FROM orders WHERE order_number = ?");
    $stmt->bind_param("s", $orderNumber);
    $stmt->execute();
    $stmt->bind_result($name);
    if ($stmt->fetch()) {
        $orderData = json_decode($name, true);
        if (isset($orderData['domain']) && isset($orderData['tld'])) {
            $additionalData = $orderData['domain'] . '.' . $orderData['tld'];
        }
    }
    $stmt->close();
    return $additionalData;
}

// Function to retrieve user by ID
function getUserById($userId, $conn) {
    $stmt = $conn->prepare("SELECT email, name, preferred_region FROM users WHERE user_id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    return $user;
}

?>