<?php
include_once includes('config.php');
header('Content-Type: application/json');

// Create an array to hold the invoice details and order details
$response = array();

// Retrieve the selected invoice
if (isset($_POST['invoice_number'])) {
    $invoice_number = $_POST['invoice_number'];
    $user_id = $_SESSION['user']['id'];
    $sql = "SELECT * FROM invoices WHERE user_id = ? AND invoice_number = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $user_id, $invoice_number);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        $invoice = $result->fetch_assoc();

        // Add invoice details to the response array
        $response['invoice_status'] = $invoice['invoice_status'];
        $response['invoice_number'] = $invoice['invoice_number'];
        $response['total_amount'] = $invoice['total_amount'];
        $response['currency'] = getCurrencySign($invoice['currency']);
        $response['tax_percent'] = $invoice['tax_percent'];
        $response['tax_amount'] = $invoice['tax_amount'];
        $response['amount'] = $invoice['amount'];

        // Format the date in a human-readable and business-level format
        $invoiceDate = new DateTime($invoice['invoice_date']);
        $formattedDate = $invoiceDate->format('F j, Y');
        $response['invoice_date'] = $formattedDate;

        // Retrieve order details related to this invoice number
        $orderNumbers = json_decode($invoice['order_numbers'], true); // Parse as an array
        $orders = getOrderDetails($orderNumbers, $conn);

        if ($orders) {
            // Initialize an empty array to store all orders
            $response['orders'] = array();

            // Loop through the retrieved orders and append them to the array
            foreach ($orders as $order) {
                $product_id = $order['product_id'];
                $product = getProductDetails($product_id, $conn);

                // Add the product name to the order
                $order['product_name'] = $product['name'];


                if ($order['product_id'] === '16') {
                    // $order['name'] is a JSON string, so you need to decode it into an array
                    $nameArray = json_decode($order['name'], true);
                    // Access the values
                    $order['order_name'] = $nameArray['domain'] . '.' . $nameArray['ltd'];
                }
                $order['vat'] = '20.00';
                $currency = $order['currency'];
                // Calculate the tax amount
                $taxAmount = ($order['total_amount'] * $order['vat']) / 100;
                $formattedTaxAmount = number_format($taxAmount, 2);

                // Calculate the total amount with tax and format to two decimal places
                $vatTotalAmount = $order['total_amount'] + $taxAmount;
                $formattedVatTotalAmount = number_format($vatTotalAmount, 2);

                if ($order['quantity'] > 0) {
                    // Calculate the total amount with tax and format to two decimal places
                    $vatTotalAmount = $order['total_amount'] + $taxAmount;
                    $formattedVatTotalAmount = number_format($vatTotalAmount, 2);

                    // Calculate the price per unit and format to two decimal places
                    $unitPrice = $order['total_amount'] / $order['quantity'];
                    $formattedUnitPrice = number_format($unitPrice, 2);
                } else {
                    // Handle the case where quantity is zero (or invalid)
                    $formattedVatTotalAmount = '0.00';
                    $formattedUnitPrice = '0.00';
                }
                if ($currency === 'try') {
                    // For Turkish Lira (TL), place the currency symbol on the right
                    $priceSign = getCurrencySign($currency);
                    $order['total_amount'] = $order['total_amount'] . $priceSign;
                    $order['vat_amount'] = $formattedTaxAmount . $priceSign;
                    $order['vat_total_amount'] = $formattedVatTotalAmount . $priceSign;
                    $order['price'] = $formattedUnitPrice . $priceSign;
                } else {
                    // For other currencies, place the currency symbol on the left
                    $priceSign = getCurrencySign($currency);
                    $order['total_amount'] = $priceSign . $order['total_amount'];
                    $order['vat_amount'] = $priceSign . $formattedTaxAmount;
                    $order['vat_total_amount'] = $priceSign . $formattedVatTotalAmount;
                    $order['price'] = $priceSign . $formattedUnitPrice;
                }
                $response['orders'][] = $order;
            }
        } else {
            echo 'No orders found.';
        }

    } else {
        $response['error'] = 'Invoice not found.';
    }
} else {
    $response['error'] = 'Invoice number not provided.';
}

echo json_encode($response);

function getCurrencySign($currency) {
    $currencySigns = [
        'usd' => '$',
        'eur' => '€',
        'gbp' => '£',
        'try' => ' TL',
    ];

    return $currencySigns[$currency] ?? '$';
}

function getOrderDetails($orderNumbers, $conn) {
    // Create a comma-separated string of order numbers for the SQL query
    $orderNumberString = implode(',', array_map(function ($orderNumber) use ($conn) {
        return "'" . $conn->real_escape_string($orderNumber) . "'";
    }, $orderNumbers));

    // Query the database for order details
    $sql = "SELECT * FROM orders WHERE order_number IN ($orderNumberString)";
    $result = $conn->query($sql);

    if ($result) {
        $orders = array();

        while ($row = $result->fetch_assoc()) {
            // Add each order's details to the array
            $orders[] = $row;
        }

        return $orders;
    } else {
        return null;
    }
}

function getProductDetails($product_id, $conn) {
    // Determine the language (session) and currency (session) settings
    $country = isset($_SESSION['country']) ? $_SESSION['country'] : 'en';
    $currency = isset($_SESSION['currency']) ? strtolower($_SESSION['currency']) : 'usd'; // Ensure currency code is in lowercase

    // Define the columns based on the language and currency settings
    $nameColumn = ($country === 'tr') ? 'name_tr' : 'name';
    $descriptionColumn = ($country === 'tr') ? 'description_tr' : 'description';
    $featuresColumn = ($country === 'tr') ? 'features_tr' : 'features';
    $priceColumn = 'price_' . $currency; // Ensure currency code is in lowercase

    // Query to fetch product data
    $query = "SELECT product_id, $nameColumn AS name, $descriptionColumn AS description, $featuresColumn AS features, picture, $priceColumn AS price, category FROM products WHERE product_id = ?";

    // Use prepared statements to prevent SQL injection
    $stmt = mysqli_prepare($conn, $query);

    if ($stmt) {
        // Bind the product_id parameter
        mysqli_stmt_bind_param($stmt, "i", $product_id); // Assuming product_id is an integer

        // Execute the prepared statement
        mysqli_stmt_execute($stmt);

        // Get the result set
        $result = mysqli_stmt_get_result($stmt);

        // Fetch the product data
        $product = mysqli_fetch_assoc($result);

        // Close the statement
        mysqli_stmt_close($stmt);

        if ($product) {
            // Add the currency to the product information
            $product['currency'] = getCurrencySign($currency);

            return $product;
        } else {
            // Product not found
            return null;
        }
    } else {
        // Handle the database query error
        return null;
    }
}
